/*
 * Copyright (C) 2012-2017 Japan Smartphone Security Association
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.jssec.android.privacypolicynopreconfirm;

import java.io.IOException;
import org.json.JSONException;
import org.json.JSONObject;
import org.jssec.android.privacypolicynopreconfirm.MainActivity;
import org.jssec.android.privacypolicynopreconfirm.R;
import org.jssec.android.privacypolicynopreconfirm.ConfirmFragment.DialogListener;


import android.os.AsyncTask;
import android.os.Bundle;
import android.content.Intent;
import android.content.SharedPreferences;
import android.content.pm.PackageInfo;
import android.content.pm.PackageManager;
import android.content.pm.PackageManager.NameNotFoundException;
import android.support.v4.app.FragmentActivity;
import android.support.v4.app.FragmentManager;
import android.telephony.TelephonyManager;
import android.text.Editable;
import android.text.TextWatcher;
import android.view.Menu;
import android.view.MenuItem;
import android.view.View;
import android.widget.TextView;
import android.widget.Toast;

public class MainActivity extends FragmentActivity implements DialogListener {
	private final String BASE_URL = "https://www.example.com/pp";
	private final String GET_ID_URI = BASE_URL + "/get_id.php";
	private final String SEND_DATA_URI = BASE_URL + "/send_data.php";
	private final String DEL_ID_URI = BASE_URL + "/del_id.php";

	private final String ID_KEY = "id";
	private final String NICK_NAME_KEY = "nickname";
	private final String IMEI_KEY = "imei";

	private final String PRIVACY_POLICY_AGREED_KEY = "privacyPolicyAgreed";

	private final String PRIVACY_POLICY_PREF_NAME = "privacypolicy_preference";

	private String UserId = "";

	private final int DIALOG_TYPE_COMPREHENSIVE_AGREEMENT = 1;

	private final int VERSION_TO_SHOW_COMPREHENSIVE_AGREEMENT_ANEW = 1;

	private TextWatcher watchHandler = new TextWatcher() {

		@Override
		public void beforeTextChanged(CharSequence s, int start, int count, int after) {
		}

		@Override
		public void onTextChanged(CharSequence s, int start, int before, int count) {
			boolean buttonEnable = (s.length() > 0);

			MainActivity.this.findViewById(R.id.buttonStart).setEnabled(buttonEnable);
		}

		@Override
		public void afterTextChanged(Editable s) {
		}
	};

	@Override
	protected void onCreate(Bundle savedInstanceState) {
		super.onCreate(savedInstanceState);
		setContentView(R.layout.activity_main);

        // Fetch user ID from serverFetch user ID from server
		new GetDataAsyncTask().execute();

		findViewById(R.id.buttonStart).setEnabled(false);
		((TextView) findViewById(R.id.editTextNickname)).addTextChangedListener(watchHandler);
	}

	@Override
	protected void onStart() {
		super.onStart();

		SharedPreferences pref = getSharedPreferences(PRIVACY_POLICY_PREF_NAME, MODE_PRIVATE);
		int privacyPolicyAgreed = pref.getInt(PRIVACY_POLICY_AGREED_KEY, -1);

		if (privacyPolicyAgreed <= VERSION_TO_SHOW_COMPREHENSIVE_AGREEMENT_ANEW) {
            // *** POINT 1 *** On first launch (or application update), obtain broad consent to transmit user data that will be handled by the application.
            // When the application is updated, it is only necessary to renew the user's grant of broad consent if the updated application will handle new types of user data.
			ConfirmFragment dialog = ConfirmFragment.newInstance(R.string.privacyPolicy, R.string.agreePrivacyPolicy, DIALOG_TYPE_COMPREHENSIVE_AGREEMENT);
			dialog.setDialogListener(this);
			FragmentManager fragmentManager = getSupportFragmentManager();
			dialog.show(fragmentManager, "dialog");
		}
	}

	public void onSendToServer(View view) {
		String nickname = ((TextView) findViewById(R.id.editTextNickname)).getText().toString();
		TelephonyManager tm = (TelephonyManager) getSystemService(TELEPHONY_SERVICE);
		String imei = tm.getDeviceId();
		Toast.makeText(MainActivity.this, this.getClass().getSimpleName() + "\n - nickname : " + nickname + ", imei = " + imei, Toast.LENGTH_SHORT).show();
		new SendDataAsyncTack().execute(SEND_DATA_URI, UserId, nickname, imei);
	}

	public void onPositiveButtonClick(int type) {
		if (type == DIALOG_TYPE_COMPREHENSIVE_AGREEMENT) {
            // *** POINT 1 *** On first launch (or application update), obtain broad consent to transmit user data that will be handled by the application.
			SharedPreferences.Editor pref = getSharedPreferences(PRIVACY_POLICY_PREF_NAME, MODE_PRIVATE).edit();
			pref.putInt(PRIVACY_POLICY_AGREED_KEY, getVersionCode());
			pref.apply();
		}
	}

	public void onNegativeButtonClick(int type) {
		if (type == DIALOG_TYPE_COMPREHENSIVE_AGREEMENT) {
            // *** POINT 2 *** If the user does not grant general consent, do not transmit user data.
            // In this sample application we terminate the application in this case.
			finish();
		}
	}

	private int getVersionCode() {
		int versionCode = -1;
		PackageManager packageManager = this.getPackageManager();
		try {
			PackageInfo packageInfo = packageManager.getPackageInfo(this.getPackageName(), PackageManager.GET_ACTIVITIES);
			versionCode = packageInfo.versionCode;
		} catch (NameNotFoundException e) {
            // This is sample, so omit the exception process
		}

		return versionCode;
	}

	@Override
	public boolean onCreateOptionsMenu(Menu menu) {
		getMenuInflater().inflate(R.menu.main, menu);
		return true;
	}

	@Override
	public boolean onOptionsItemSelected(MenuItem item) {
		switch (item.getItemId()) {
		case R.id.action_show_pp:
            // *** POINT 3 *** Provide methods by which the user can review the application privacy policy.
			Intent intent = new Intent();
			intent.setClass(this, WebViewAssetsActivity.class);
			startActivity(intent);
			return true;
		case R.id.action_del_id:
            // *** POINT 4 *** Provide methods by which transmitted data can be deleted by user operations.
			new SendDataAsyncTack().execute(DEL_ID_URI, UserId);
			return true;
		case R.id.action_donot_send_id:
            // *** POINT 5 *** Provide methods by which transmitting data can be stopped by user operations.

            // If the user stop sending data, user consent is deemed to have been revoked.
			SharedPreferences.Editor pref = getSharedPreferences(PRIVACY_POLICY_PREF_NAME, MODE_PRIVATE).edit();
			pref.putInt(PRIVACY_POLICY_AGREED_KEY, 0);
			pref.apply();

            // In this sample application if the user data cannot be sent by user operations,
            // finish the application because we do nothing.
			String message  = getString(R.string.stopSendUserData);
			Toast.makeText(MainActivity.this, this.getClass().getSimpleName() + " - " + message, Toast.LENGTH_SHORT).show();
			finish();

			return true;		}
		return false;
	}

	private class GetDataAsyncTask extends AsyncTask<String, Void, String> {
		private String extMessage = "";

		@Override
		protected String doInBackground(String... params) {
            // *** POINT 6 *** Use UUIDs or cookies to keep track of user data
            // In this sample we use an ID generated on the server side
			SharedPreferences sp = getSharedPreferences(PRIVACY_POLICY_PREF_NAME, MODE_PRIVATE);
			UserId = sp.getString(ID_KEY, null);
			if (UserId == null) {
                // No token in SharedPreferences; fetch ID from server
				try {
					UserId = NetworkUtil.getCookie(GET_ID_URI, "", "id");
				} catch (IOException e) {
                    // Catch exceptions such as certification errors
					extMessage = e.toString();
				}

                // Store the fetched ID in SharedPreferences
				sp.edit().putString(ID_KEY, UserId).commit();
			}
			return UserId;
		}

		@Override
		protected void onPostExecute(final String data) {
			String status = (data != null) ? "success" : "error";
			Toast.makeText(MainActivity.this, this.getClass().getSimpleName() + " - " + status + " : " + extMessage, Toast.LENGTH_SHORT).show();
		}
	}

	private class SendDataAsyncTack extends AsyncTask<String, Void, Boolean> {
		private String extMessage = "";

		@Override
		protected Boolean doInBackground(String... params) {
			String url = params[0];
			String id = params[1];
			String nickname = params.length > 2 ? params[2] : null;
			String imei = params.length > 3 ? params[3] : null;

			Boolean result = false;
			try {
				JSONObject jsonData = new JSONObject();
				jsonData.put(ID_KEY, id);

				if (nickname != null)
					jsonData.put(NICK_NAME_KEY, nickname);

				if (imei != null)
					jsonData.put(IMEI_KEY, imei);

				NetworkUtil.sendJSON(url, "", jsonData.toString());

				result = true;
			} catch (IOException e) {
                // Catch exceptions such as certification errors
				extMessage = e.toString();
			} catch (JSONException e) {
				extMessage = e.toString();
			}
			return result;
		}

		@Override
		protected void onPostExecute(Boolean result) {
			String status = result ? "Success" : "Error";
			Toast.makeText(MainActivity.this, this.getClass().getSimpleName() + " - " + status + " : " + extMessage, Toast.LENGTH_SHORT).show();
		}
	}
}
